/*
 * Decompiled with CFR 0.152.
 */
package ru.minezone.launcher.download;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.json.JSONArray;
import org.json.JSONObject;
import ru.minezone.launcher.MineZoneLauncher;
import ru.minezone.launcher.client.ClientProfile;
import ru.minezone.launcher.utils.LogHelper;
import ru.minezone.launcher.utils.OSHelper;

public class DownloadManager {
    private static final int BUFFER_SIZE = 8192;
    private static final int DOWNLOAD_THREADS = 4;
    private static final int CONNECT_TIMEOUT = 15000;
    private static final int READ_TIMEOUT = 30000;
    private static final Set<String> SYNC_FOLDERS = new HashSet<String>(Arrays.asList("mods", "mods/1.7.10", "config", "scripts"));
    private static final Set<String> SYNC_EXTENSIONS = new HashSet<String>(Arrays.asList(".jar", ".zip", ".cfg", ".conf", ".json", ".zs"));
    private ExecutorService executor;
    private volatile boolean cancelled = false;
    private Consumer<DownloadProgress> progressCallback;
    private Set<String> disabledMods = new HashSet<String>();
    private AtomicLong bytesDownloadedLastSecond = new AtomicLong(0L);
    private volatile double currentSpeedMbps = 0.0;
    private long lastSpeedUpdate = 0L;

    public DownloadManager() {
        this.executor = Executors.newFixedThreadPool(4);
    }

    public boolean downloadClient(ClientProfile profile, Consumer<DownloadProgress> callback) {
        this.progressCallback = callback;
        this.cancelled = false;
        try {
            File workDir = MineZoneLauncher.getInstance().getWorkDir();
            File clientDir = new File(workDir, "clients/" + profile.getClientDirName());
            this.scanDisabledMods(clientDir);
            this.reportProgress("\u0417\u0430\u0433\u0440\u0443\u0437\u043a\u0430 \u043a\u043b\u0438\u0435\u043d\u0442\u0430 " + profile.getTitle() + "...", 0, 0, 0);
            if (!this.downloadClientFiles(profile, workDir, callback)) {
                return false;
            }
            if (this.cancelled) {
                return false;
            }
            this.reportProgress("\u0417\u0430\u0433\u0440\u0443\u0437\u043a\u0430 \u0440\u0435\u0441\u0443\u0440\u0441\u043e\u0432...", 0, 0, 0);
            if (!this.downloadAssets(profile, workDir, callback)) {
                LogHelper.warning("Assets download failed, continuing...");
            }
            if (this.cancelled) {
                return false;
            }
            this.reportProgress("\u041f\u0440\u043e\u0432\u0435\u0440\u043a\u0430 Java...", 0, 0, 0);
            if (!this.downloadJavaIfNeeded(profile, workDir, callback)) {
                LogHelper.warning("Java download failed, will use system Java");
            }
            this.reportProgress("\u0413\u043e\u0442\u043e\u0432\u043e!", 100, 0, 0);
            return true;
        }
        catch (Exception e) {
            LogHelper.error("Download error", e);
            this.reportProgress("\u041e\u0448\u0438\u0431\u043a\u0430: " + e.getMessage(), 0, 0, 0);
            return false;
        }
    }

    private void scanDisabledMods(File clientDir) {
        File optionalDir;
        File mods1710Dir;
        this.disabledMods.clear();
        File modsDir = new File(clientDir, "mods");
        if (modsDir.exists()) {
            this.scanDisabledInDir(modsDir);
        }
        if ((mods1710Dir = new File(clientDir, "mods/1.7.10")).exists()) {
            this.scanDisabledInDir(mods1710Dir);
        }
        if ((optionalDir = new File(modsDir, "optional")).exists()) {
            this.scanDisabledInDir(optionalDir);
        }
        if (!this.disabledMods.isEmpty()) {
            LogHelper.info("Found " + this.disabledMods.size() + " disabled mods: " + this.disabledMods);
        }
    }

    private void scanDisabledInDir(File dir) {
        File[] files = dir.listFiles((d, name) -> name.endsWith(".disabled"));
        if (files != null) {
            for (File file : files) {
                String originalName = file.getName().replace(".disabled", "");
                this.disabledMods.add(originalName);
                LogHelper.info("Found disabled mod: " + originalName);
            }
        }
    }

    private boolean isModDisabled(String fileName) {
        return this.disabledMods.contains(fileName) || this.disabledMods.contains(fileName + ".jar");
    }

    private void synchronizeFolders(File clientDir, List<FileEntry> serverFiles) {
        Set serverPaths = serverFiles.stream().map(f -> f.path.toLowerCase().replace("\\", "/")).collect(Collectors.toSet());
        HashSet serverPathsWithDisabled = new HashSet(serverPaths);
        for (String path : serverPaths) {
            serverPathsWithDisabled.add(path + ".disabled");
        }
        int deletedCount = 0;
        for (String syncFolder : SYNC_FOLDERS) {
            File[] files;
            File folder = new File(clientDir, syncFolder);
            if (!folder.exists() || !folder.isDirectory() || (files = folder.listFiles()) == null) continue;
            for (File file : files) {
                if (file.isDirectory()) continue;
                String fileName = file.getName().toLowerCase();
                boolean shouldSync = false;
                for (String ext : SYNC_EXTENSIONS) {
                    if (!fileName.endsWith(ext) && !fileName.endsWith(ext + ".disabled")) continue;
                    shouldSync = true;
                    break;
                }
                if (!shouldSync) continue;
                String relativePath = syncFolder + "/" + file.getName();
                relativePath = relativePath.toLowerCase().replace("\\", "/");
                String baseRelativePath = relativePath.replace(".disabled", "");
                if (serverPathsWithDisabled.contains(relativePath) || serverPathsWithDisabled.contains(baseRelativePath)) continue;
                LogHelper.info("Removing extra file: " + relativePath);
                if (file.delete()) {
                    ++deletedCount;
                    continue;
                }
                LogHelper.warning("Failed to delete: " + file.getAbsolutePath());
            }
        }
        if (deletedCount > 0) {
            LogHelper.info("Removed " + deletedCount + " extra files during sync");
        }
    }

    private boolean downloadClientFiles(ClientProfile profile, File workDir, Consumer<DownloadProgress> callback) {
        try {
            List<FileEntry> serverFiles;
            File clientDir = new File(workDir, "clients/" + profile.getClientDirName());
            if (!clientDir.exists()) {
                clientDir.mkdirs();
            }
            if ((serverFiles = this.fetchFileList(profile.getClientDirName())) == null || serverFiles.isEmpty()) {
                LogHelper.warning("No files list for client, checking if already downloaded");
                File minecraftJar = new File(clientDir, "minecraft.jar");
                File clientJar = new File(clientDir, "client.jar");
                return minecraftJar.exists() || clientJar.exists();
            }
            LogHelper.info("Client has " + serverFiles.size() + " files");
            this.reportProgress("\u0421\u0438\u043d\u0445\u0440\u043e\u043d\u0438\u0437\u0430\u0446\u0438\u044f \u0444\u0430\u0439\u043b\u043e\u0432...", 0, 0, 0);
            this.synchronizeFolders(clientDir, serverFiles);
            List<FileEntry> filesToDownload = this.checkFiles(clientDir, serverFiles);
            if (this.cancelled) {
                return false;
            }
            if (filesToDownload.isEmpty()) {
                LogHelper.info("All client files are up to date");
                return true;
            }
            long totalSize = filesToDownload.stream().mapToLong(f -> f.size).sum();
            return this.downloadFiles(clientDir, filesToDownload, totalSize, "\u041a\u043b\u0438\u0435\u043d\u0442");
        }
        catch (Exception e) {
            LogHelper.error("Client download error", e);
            return false;
        }
    }

    private boolean downloadAssets(ClientProfile profile, File workDir, Consumer<DownloadProgress> callback) {
        try {
            List<FileEntry> filesToDownload;
            String assetsFolder = "assets" + profile.getAssetIndex();
            File assetsDir = new File(workDir, "clients/" + assetsFolder);
            File clientAssetsDir = new File(workDir, "clients/" + profile.getClientDirName() + "/assets");
            List<FileEntry> serverFiles = this.fetchFileList(assetsFolder);
            if (serverFiles == null || serverFiles.isEmpty()) {
                LogHelper.info("No separate assets folder, checking client folder");
                return true;
            }
            if (!assetsDir.exists()) {
                assetsDir.mkdirs();
            }
            if ((filesToDownload = this.checkFiles(assetsDir, serverFiles)).isEmpty()) {
                LogHelper.info("All assets are up to date");
            } else {
                long totalSize = filesToDownload.stream().mapToLong(f -> f.size).sum();
                if (!this.downloadFiles(assetsDir, filesToDownload, totalSize, "\u0420\u0435\u0441\u0443\u0440\u0441\u044b")) {
                    return false;
                }
            }
            if (!clientAssetsDir.exists() && assetsDir.exists()) {
                try {
                    Files.createSymbolicLink(clientAssetsDir.toPath(), assetsDir.toPath(), new FileAttribute[0]);
                }
                catch (Exception e) {
                    LogHelper.info("Cannot create symlink, assets will be used from shared folder");
                }
            }
            return true;
        }
        catch (Exception e) {
            LogHelper.error("Assets download error", e);
            return false;
        }
    }

    private boolean downloadJavaIfNeeded(ClientProfile profile, File workDir, Consumer<DownloadProgress> callback) {
        try {
            long totalSize;
            List<FileEntry> filesToDownload;
            String javaVersion = this.getRequiredJavaVersion(profile.getVersion());
            String javaFolder = "jre" + javaVersion;
            File javaDir = new File(workDir, "clients/" + javaFolder);
            File javaBin = new File(javaDir, OSHelper.isWindows() ? "bin/java.exe" : "bin/java");
            if (javaBin.exists()) {
                LogHelper.info("Java " + javaVersion + " already exists");
                profile.setJavaPath(javaBin);
                return true;
            }
            List<FileEntry> serverFiles = this.fetchFileList(javaFolder);
            if (serverFiles == null || serverFiles.isEmpty()) {
                LogHelper.info("No bundled Java " + javaVersion + " on server, will use system Java");
                return true;
            }
            this.reportProgress("\u0417\u0430\u0433\u0440\u0443\u0437\u043a\u0430 Java " + javaVersion + "...", 0, 0, 0);
            if (!javaDir.exists()) {
                javaDir.mkdirs();
            }
            if (!(filesToDownload = this.checkFiles(javaDir, serverFiles)).isEmpty() && !this.downloadFiles(javaDir, filesToDownload, totalSize = filesToDownload.stream().mapToLong(f -> f.size).sum(), "Java " + javaVersion)) {
                return false;
            }
            if (!OSHelper.isWindows() && javaBin.exists()) {
                javaBin.setExecutable(true);
            }
            profile.setJavaPath(javaBin);
            LogHelper.info("Java " + javaVersion + " ready");
            return true;
        }
        catch (Exception e) {
            LogHelper.error("Java download error", e);
            return false;
        }
    }

    private String getRequiredJavaVersion(String mcVersion) {
        if (mcVersion.startsWith("1.7") || mcVersion.startsWith("1.8") || mcVersion.startsWith("1.9") || mcVersion.startsWith("1.10") || mcVersion.startsWith("1.11") || mcVersion.startsWith("1.12") || mcVersion.startsWith("1.13") || mcVersion.startsWith("1.14") || mcVersion.startsWith("1.15") || mcVersion.startsWith("1.16")) {
            return "8";
        }
        if (mcVersion.startsWith("1.17")) {
            return "16";
        }
        return "17";
    }

    private List<FileEntry> fetchFileList(String folderName) {
        try {
            String url = "https://minezone.su/launcher/files/clients/" + folderName + "/files.json";
            LogHelper.info("Fetching file list from: " + url);
            HttpURLConnection conn = (HttpURLConnection)new URL(url).openConnection();
            conn.setConnectTimeout(15000);
            conn.setReadTimeout(30000);
            conn.setRequestProperty("User-Agent", "MineZoneLauncher/1.0.0");
            if (conn.getResponseCode() != 200) {
                LogHelper.warning("Server returned: " + conn.getResponseCode() + " for " + folderName);
                return null;
            }
            StringBuilder response = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(conn.getInputStream(), "UTF-8"));){
                String line;
                while ((line = reader.readLine()) != null) {
                    response.append(line);
                }
            }
            return this.parseFileList(response.toString(), folderName);
        }
        catch (Exception e) {
            LogHelper.warning("Failed to fetch file list for " + folderName + ": " + e.getMessage());
            return null;
        }
    }

    private List<FileEntry> parseFileList(String json, String folderName) {
        ArrayList<FileEntry> files = new ArrayList<FileEntry>();
        try {
            JSONObject root = new JSONObject(json);
            if (root.has("files")) {
                JSONArray filesArray = root.getJSONArray("files");
                for (int i = 0; i < filesArray.length(); ++i) {
                    JSONObject file = filesArray.getJSONObject(i);
                    files.add(this.parseFileEntry(file, folderName));
                }
            } else if (root.has("entries")) {
                this.parseHashedDir(root.getJSONObject("entries"), "", files, folderName);
            } else {
                this.parseHashedDir(root, "", files, folderName);
            }
        }
        catch (Exception e) {
            LogHelper.error("Failed to parse file list", e);
        }
        return files;
    }

    private void parseHashedDir(JSONObject dir, String path, List<FileEntry> files, String folderName) {
        for (String key : dir.keySet()) {
            Object value = dir.get(key);
            if (!(value instanceof JSONObject)) continue;
            JSONObject obj = (JSONObject)value;
            if (obj.has("size")) {
                FileEntry entry = new FileEntry();
                entry.name = key;
                entry.path = path.isEmpty() ? key : path + "/" + key;
                entry.size = obj.getLong("size");
                entry.hash = obj.optString("hash", obj.optString("md5", ""));
                entry.hashType = obj.optString("hashType", "MD5");
                entry.url = "https://minezone.su/launcher/files/clients/" + folderName + "/" + entry.path;
                entry.type = this.detectFileType(entry.path, entry.name);
                files.add(entry);
                continue;
            }
            String newPath = path.isEmpty() ? key : path + "/" + key;
            this.parseHashedDir(obj, newPath, files, folderName);
        }
    }

    private FileEntry parseFileEntry(JSONObject obj, String folderName) {
        FileEntry entry = new FileEntry();
        entry.name = obj.getString("name");
        entry.path = obj.optString("path", entry.name);
        entry.size = obj.getLong("size");
        entry.hash = obj.optString("hash", obj.optString("md5", ""));
        entry.hashType = obj.optString("hashType", "MD5");
        entry.type = obj.optString("type", "OTHER");
        entry.url = obj.optString("url", "https://minezone.su/launcher/files/clients/" + folderName + "/" + entry.path);
        return entry;
    }

    private String detectFileType(String path, String name) {
        path = path.toLowerCase();
        name = name.toLowerCase();
        if (path.startsWith("libraries") || path.contains("/libraries/")) {
            return "LIBRARY";
        }
        if (path.startsWith("natives") || path.contains("/natives/")) {
            return "NATIVE";
        }
        if (path.startsWith("mods") || path.contains("/mods/")) {
            return "MOD";
        }
        if (path.startsWith("config") || path.contains("/config/")) {
            return "CONFIG";
        }
        if (path.startsWith("assets") || path.contains("/assets/")) {
            return "ASSET";
        }
        if (name.equals("minecraft.jar") || name.equals("client.jar")) {
            return "CLIENT";
        }
        if (name.endsWith(".jar") && path.contains("forge")) {
            return "FORGE";
        }
        return "OTHER";
    }

    private List<FileEntry> checkFiles(File baseDir, List<FileEntry> serverFiles) {
        ArrayList<FileEntry> toDownload = new ArrayList<FileEntry>();
        int checked = 0;
        int total = serverFiles.size();
        int skippedDisabled = 0;
        for (FileEntry entry : serverFiles) {
            String localHash;
            File disabledFile;
            if (this.cancelled) break;
            if (++checked % 100 == 0) {
                int percent = (int)((double)checked * 100.0 / (double)total);
                this.reportProgress("\u041f\u0440\u043e\u0432\u0435\u0440\u043a\u0430: " + checked + "/" + total, percent, checked, total);
            }
            File localFile = new File(baseDir, entry.path);
            if (entry.type != null && entry.type.equals("MOD") && this.isModDisabled(entry.name) && (disabledFile = new File(baseDir, entry.path + ".disabled")).exists()) {
                LogHelper.info("Skipping disabled mod: " + entry.name);
                ++skippedDisabled;
                continue;
            }
            File disabledVersion = new File(baseDir, entry.path + ".disabled");
            if (disabledVersion.exists() && entry.type != null && entry.type.equals("MOD")) {
                LogHelper.info("Found disabled version, skipping: " + entry.name);
                ++skippedDisabled;
                continue;
            }
            if (!localFile.exists()) {
                toDownload.add(entry);
                continue;
            }
            if (localFile.length() != entry.size) {
                toDownload.add(entry);
                continue;
            }
            if (entry.hash == null || entry.hash.isEmpty() || entry.hash.equalsIgnoreCase(localHash = DownloadManager.calculateHash(localFile, entry.hashType))) continue;
            toDownload.add(entry);
        }
        if (skippedDisabled > 0) {
            LogHelper.info("Skipped " + skippedDisabled + " disabled mods");
        }
        return toDownload;
    }

    private boolean downloadFiles(File baseDir, List<FileEntry> files, long totalSize, String label) {
        AtomicLong downloadedBytes = new AtomicLong(0L);
        AtomicInteger downloadedFiles = new AtomicInteger(0);
        int totalFiles = files.size();
        this.bytesDownloadedLastSecond.set(0L);
        this.lastSpeedUpdate = System.currentTimeMillis();
        this.currentSpeedMbps = 0.0;
        LogHelper.info("Downloading " + totalFiles + " files for " + label);
        ArrayList<Future<Boolean>> futures = new ArrayList<Future<Boolean>>();
        for (FileEntry entry : files) {
            if (this.cancelled) break;
            Future<Boolean> future = this.executor.submit(() -> {
                try {
                    if (this.cancelled) {
                        return false;
                    }
                    File targetFile = new File(baseDir, entry.path);
                    targetFile.getParentFile().mkdirs();
                    boolean success = this.downloadFile(entry.url, targetFile, entry.size, bytes -> {
                        long total = downloadedBytes.addAndGet((long)bytes);
                        this.bytesDownloadedLastSecond.addAndGet((long)bytes);
                        long now = System.currentTimeMillis();
                        if (now - this.lastSpeedUpdate >= 1000L) {
                            long bytesInSecond = this.bytesDownloadedLastSecond.getAndSet(0L);
                            this.currentSpeedMbps = (double)bytesInSecond * 8.0 / 1000000.0;
                            this.lastSpeedUpdate = now;
                        }
                        int percent = totalSize > 0L ? (int)(total * 100L / totalSize) : 0;
                        this.reportProgress(label + ": " + entry.name, percent, downloadedFiles.get(), totalFiles, this.currentSpeedMbps);
                    });
                    if (success) {
                        downloadedFiles.incrementAndGet();
                    }
                    return success;
                }
                catch (Exception e) {
                    LogHelper.error("Failed to download: " + entry.name, e);
                    return false;
                }
            });
            futures.add(future);
        }
        boolean allSuccess = true;
        for (Future<Boolean> future : futures) {
            try {
                if (((Boolean)future.get()).booleanValue()) continue;
                allSuccess = false;
            }
            catch (Exception e) {
                allSuccess = false;
            }
        }
        return allSuccess;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean downloadFile(String urlStr, File targetFile, long expectedSize, Consumer<Long> progressCallback) {
        File tempFile = new File(targetFile.getAbsolutePath() + ".tmp");
        try {
            URL url = new URL(urlStr);
            HttpURLConnection conn = (HttpURLConnection)url.openConnection();
            conn.setConnectTimeout(15000);
            conn.setReadTimeout(30000);
            conn.setRequestProperty("User-Agent", "MineZoneLauncher/1.0.0");
            int responseCode = conn.getResponseCode();
            if (responseCode != 200) {
                LogHelper.error("HTTP " + responseCode + " for: " + urlStr);
                return false;
            }
            try (InputStream in = conn.getInputStream();
                 FileOutputStream out = new FileOutputStream(tempFile);){
                int bytesRead;
                byte[] buffer = new byte[8192];
                while ((bytesRead = in.read(buffer)) != -1) {
                    if (this.cancelled) {
                        tempFile.delete();
                        boolean bl = false;
                        return bl;
                    }
                    out.write(buffer, 0, bytesRead);
                    if (progressCallback == null) continue;
                    progressCallback.accept(Long.valueOf(bytesRead));
                }
            }
            if (expectedSize > 0L && tempFile.length() != expectedSize) {
                LogHelper.error("Size mismatch: expected " + expectedSize + ", got " + tempFile.length());
                tempFile.delete();
                return false;
            }
            if (targetFile.exists()) {
                targetFile.delete();
            }
            Files.move(tempFile.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            return true;
        }
        catch (Exception e) {
            LogHelper.error("Download failed: " + urlStr, e);
            tempFile.delete();
            return false;
        }
    }

    public static String calculateHash(File file, String algorithm) {
        try {
            MessageDigest digest = MessageDigest.getInstance(algorithm.replace("-", ""));
            try (FileInputStream fis = new FileInputStream(file);){
                int bytesRead;
                byte[] buffer = new byte[8192];
                while ((bytesRead = fis.read(buffer)) != -1) {
                    digest.update(buffer, 0, bytesRead);
                }
            }
            byte[] hash = digest.digest();
            StringBuilder sb = new StringBuilder();
            for (byte b : hash) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        }
        catch (Exception e) {
            return "";
        }
    }

    public void cancel() {
        this.cancelled = true;
    }

    public void shutdown() {
        this.cancelled = true;
        this.executor.shutdownNow();
    }

    private void reportProgress(String status, int percent, int current, int total) {
        this.reportProgress(status, percent, current, total, 0.0);
    }

    private void reportProgress(String status, int percent, int current, int total, double speedMbps) {
        if (this.progressCallback != null) {
            this.progressCallback.accept(new DownloadProgress(status, percent, current, total, speedMbps));
        }
    }

    public static class DownloadProgress {
        public final String status;
        public final int percent;
        public final int currentFile;
        public final int totalFiles;
        public final double speedMbps;

        public DownloadProgress(String status, int percent, int currentFile, int totalFiles) {
            this(status, percent, currentFile, totalFiles, 0.0);
        }

        public DownloadProgress(String status, int percent, int currentFile, int totalFiles, double speedMbps) {
            this.status = status;
            this.percent = percent;
            this.currentFile = currentFile;
            this.totalFiles = totalFiles;
            this.speedMbps = speedMbps;
        }

        public String getFormattedSpeed() {
            if (this.speedMbps <= 0.0) {
                return "";
            }
            if (this.speedMbps < 1.0) {
                return String.format("%.0f \u041a\u0431\u0438\u0442/\u0441", this.speedMbps * 1000.0);
            }
            return String.format("%.1f \u041c\u0431\u0438\u0442/\u0441", this.speedMbps);
        }
    }

    public static class FileEntry {
        public String name;
        public String path;
        public long size;
        public String hash;
        public String hashType = "MD5";
        public String type;
        public String url;
    }
}

